<?php

/**
 * Yo Payment Service
 *
 * @package     Ridein
 * @subpackage  Services\Payments
 * @category    Yo
 * @author      Sourcemonster Team
 * @version     3.8.0
 * @link        https://sourcemonster.in
 */

namespace App\Services\PaymentMethods;


use App\Contracts\PaymentInterface;
use Illuminate\Support\Facades\Log;
use YoAPI;


class YoPayment implements PaymentInterface
{

    private $yoAPI;

    /**
     * Intialize Stripe with Secret key
     *
     */
    public function __construct()
    {
        $api_username =  payment_gateway('username', 'Yo');
        $api_password = payment_gateway('password', 'Yo');
        $mode = payment_gateway('mode', 'Yo');

        $this->yoAPI = new YoAPI($api_username, $api_password, $mode);
    }

    public function makePayment($payment_data, $indent_id = '')
    {
        try {


            // Create a unique transaction reference that you will reference this payment with
            $transaction_reference = date("YmdHis") . rand(1, 1000);
            $this->yoAPI->set_external_reference($transaction_reference);

            // Set nonblocking to TRUE so that you get an instant response
            $this->yoAPI->set_nonblocking("TRUE");

            // Log information about the initiated payment
            Log::info("Initiating Yo Payment for order {$payment_data["user_id"]} with transaction reference {$transaction_reference}");
            $reason = ucfirst($payment_data['pay_for']);
            $phone_number = $payment_data['user_country_code'] . $payment_data['user_phone'];
            $response = $this->yoAPI->ac_deposit_funds($phone_number, $payment_data['amount'], $reason);

            // Log the Yo Payments API response
            Log::info("Yo Payments API Response: " . json_encode($response));

            if ($response['Status'] == 'OK') {
                // check transaction
                Log::info("Payment initiated successfully. Checking transaction status.");
                return $this->checkTransaction($response['TransactionReference']);   // Save this transaction for future reference
            } else {
                $errorMessage = "Yo Payments Error: " . $response['StatusMessage'];

                // Log the error
                Log::error($errorMessage);
                return arrayToObject(array(
                    'status'         => false,
                    'status_message' =>   $errorMessage,
                ));
            }
        } catch (\Exception $e) {
            Log::error(json_encode($e));
            return arrayToObject(array(
                'status'         => false,
                'status_message' => $e->getMessage(),
            ));
        }
    }


    public function checkTransaction($transaction_reference)
    {
        // Log information about checking transaction status
        Log::info("Checking Yo Payment transaction status for order {$transaction_reference} with transaction reference {$transaction_reference}");

        $maxAttempts = 10; // You can adjust the maximum number of attempts
        $attempts = 0;

        do {
            $transaction = $this->yoAPI->ac_transaction_check_status($transaction_reference);
            $attempts++;

            // Log the Yo Payments API response for transaction status check
            Log::info("Yo Payments Transaction Status Check Response: " . json_encode($transaction));

            if (strcmp($transaction['TransactionStatus'], 'SUCCEEDED') == 0) {

                return arrayToObject(array(
                    'status'         => true,
                    'status_message' => 'payment successful',
                    'pending_status' => false,
                    'transaction_id' => $transaction['MNOTransactionReferenceId']
                ));
            } elseif (strcmp($transaction['TransactionStatus'], 'DECLINED') == 0) {
                Log::info("Transaction for order {$transaction_reference} declined. Redirecting to payment fail route.");
                return arrayToObject(array(
                    'status'         => false,
                    'status_message' => 'Transaction declined',
                    'pending_status' => false
                ));
            }

            // Add a delay before the next attempt (adjust as needed)
            sleep(5);
        } while ($attempts < $maxAttempts);

        // If the loop completes without a final status, handle it accordingly
        Log::info("Transaction for order {$transaction_reference} is still pending after {$maxAttempts} attempts. Redirecting to payment fail route.");
        return arrayToObject(array(
            'status'         => false,
            'status_message' => 'Transaction is still pending',
            'pending_status' => true
        ));
    }
}
